package gov.va.med.mhv.rxrefill.service.validator;

import gov.va.med.mhv.common.api.util.ResponseUtil;
import gov.va.med.mhv.rxrefill.dto.MedicationHistoryDTO;
import gov.va.med.mhv.rxrefill.util.RxMessagesUtil;

import java.util.Date;
import java.util.LinkedHashMap;

import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

@Component 
public class MedicationsValidator {
	
	private static final int NAME_MIN_LENGTH = 3;
	private static final int NAME_MAX_LENGTH = 50;
	private static final int COMMENTS_MAX_LENGTH = 150;

	@Autowired
	private RxMessagesUtil messagesUtil;
	
	protected void validateCategory(MedicationHistoryDTO medication, LinkedHashMap<String, String> validationErrors) {
		String value = null;
		if (medication.getCategory() != null && medication.getCategory().trim().length()>0) {
			value = medication.getCategory();
		}
		//check requiredness:
		if( StringUtils.isEmpty(value)) {
			Validations.validateRequired("Category", value, validationErrors);
						
			//no need to continue:
			return;
		}		
	}

	protected void validateName(MedicationHistoryDTO medication, LinkedHashMap<String, String> validationErrors) {
		String value = null;
		if (medication.getMedicationName() != null) {
			value = StringUtils.trimToNull(medication.getMedicationName());
		}
		//check requiredness:		
		if( value == null ) {
			Validations.validateRequired("Name", value, validationErrors);
						
			//no need to continue:
			return;
		}
		
		Validations.validateMinLength("Name", value, NAME_MIN_LENGTH, validationErrors);
		Validations.validateMaxLength("Name", value, NAME_MAX_LENGTH, validationErrors);
	}	
	
	protected void validateStartDate(MedicationHistoryDTO medication, LinkedHashMap<String, String> validationErrors) {
		Date value = medication.getStartDate();
		//check for a value to validate:
		if ( value == null ) {
			Validations.validateRequired("Start Date", value, validationErrors);
			//nothing to validate
			return;
		}
	}	
	
	/**
	 * Ensure that the start date is before the end date
	 *
	 */
	private void validateDates(MedicationHistoryDTO medication, LinkedHashMap<String, String> validationErrors) {
		Date startDate = medication.getStartDate();
		Date stopDate = medication.getEndDate();
		
		if (null != startDate && null != stopDate) {
			if (!startDate.before(stopDate) && !startDate.equals(stopDate)) {
				validationErrors.put(messagesUtil.getStartDateBeforeStopDate(), messagesUtil.getStartDateBeforeStopDate());
			}
		}
	}
	
	protected void validateReason(MedicationHistoryDTO medication, LinkedHashMap<String, String> validationErrors) {
		String value = StringUtils.trimToNull(medication.getReason());
		//check for a value to validate:
		if ( value == null ) {
			//nothing to validate
			return;
		}						
		//check max length
		Validations.validateMaxLength("Reason for Taking", value, COMMENTS_MAX_LENGTH, validationErrors);
	}
	
	protected void validateComments(MedicationHistoryDTO medication, LinkedHashMap<String, String> validationErrors) {
		String value = null;
		if (medication.getComments() != null) {
			value = StringUtils.trimToNull(medication.getComments());
		}
		//check for a value to validate:
		if ( value == null ) {
			//nothing to validate
			return;
		}						
		//check max length
		Validations.validateMaxLength("Comments", value, COMMENTS_MAX_LENGTH, validationErrors);
	}
	
	/**
	 * Executes all field-level validations
	 */
	public ResponseUtil fieldValidation(MedicationHistoryDTO medication) {
		ResponseUtil response = new ResponseUtil();
		LinkedHashMap<String, String> validationErrors = new LinkedHashMap<String, String>();
		
		validateCategory(medication, validationErrors);
		validateName(medication, validationErrors);
		validateStartDate(medication, validationErrors);
		validateDates(medication, validationErrors);
		validateReason(medication, validationErrors);
		validateComments(medication, validationErrors);
		
		response.setValidationErrors(validationErrors);
		
		return response;
	}
	
}